function [gx,hx,retcode] = bfor_endogenous_ms_solver_iterativeprocedure(Pmatss,A,B,ny,nx)
% 
% performs the an iterative procedure for solving first-order peturbation
%   of MS models developed by Foerster, Rubio-Ramirez, Waggoner, and Zha 
%   (2013), adapted for endogenous switching by Benigno, Foerster, Otrok, 
%   and Rebucci (2020)
% 
% Updated 2020/12
% Benigno, Foerster, Otrok, and Rebucci
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
%  INPUTS
%   Pmatss  = (ns x ns) matrix where P(i,j)=Pr(s_{t+1}=j|s_{t}=i)
%   A       = (ns x ns) cell, where A{s,sp} is a (n x n) matrix and the 
%               first order derivatives of f with respect to [yvarsp xvars]
%   B       = (ns x ns) cell, where B{s,sp} is a (n x n) matrix and the 
%               first order derivatives of f with respect to [yvars xlag]
%   ny      = number of non-predetermined variables in the system
%   nx      = number of predetermined variables in the system
%  
%  OUTPUTS
%   gx      = (ny x nx x ns) array with the solution for the
%               non-predetermined variables, gx(i,j,s) is the response of
%               y(i) to x(j) in regime s
%   hx      = (nx x nx x ns) array with the solution for the
%               predetermined variables, hx(i,j,s) is the response of
%               y(i) to x(j) in regime s
%   retcode  = indicator taking values
%       0: solution not found, iterative procedure does not converge or
%           non-MSS solution found
%       1: solution found, iterative procedure converges and is MSS
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %



% -- Setup -- %
ns = length(Pmatss);
crit = 1e-10;
maxit = 1000;


% -- Constructing Matrices -- %
gx = cell(ns,1);
hx = cell(ns,1);

% guess
for s = 1:ns
    gx{s} = zeros(ny,nx);
    hx{s} = zeros(nx,nx);
end

jj = 0;
norm = Inf;
while norm > crit && jj < maxit
    jj = jj + 1;
    gx0 = gx;
    hx0 = hx;   
    
    % now iteratively solve
    s = 1;
    stop = 0;
    while s <= ns && stop == 0
        Amat = zeros(nx+ny,nx+ny);
        Bmat = zeros(nx+ny,nx+ny);
        for sp = 1:ns        
            if sp == s
                Amat = Amat + [Pmatss(s,sp)*A{sp,s}(:,1:nx) Pmatss(s,sp)*A{sp,s}(:,nx+1:nx+ny)];
            else
                Amat(:,1:nx) = Amat(:,1:nx) + Pmatss(s,sp)*A{sp,s}(:,1:nx) + Pmatss(s,sp)*A{sp,s}(:,nx+1:nx+ny)*gx{sp};
            end
            Bmat = Bmat - Pmatss(s,sp)*B{sp,s};
        end
        
        % eigenvalue decomposition
        if isinf(max(max(abs([Amat Bmat])))) || isnan(max(max(abs([Amat Bmat]))))
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            retcode = 0;
            warning('Infinite Matrix')
            return;
        else
            [V, D] = eig(Bmat,Amat);
        end

        %reorder eigenvalues to be in ascending order
        [~,I] = sort(abs(diag(D)));
        d = diag(D);
        D = diag(d(I));
        V = V(:,I);
        
        % solve system if a solution exists (ie invertibility is met)
        if rank(V(1:nx,1:nx)) == nx
            gx{s} = V(nx+1:nx+ny,1:nx)/V(1:nx,1:nx);
            hx{s} = V(1:nx,1:nx)*D(1:nx,1:nx)/V(1:nx,1:nx);
        else
            warning('Rank Condition Not Met - No Solution');
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            stop = 1;
        end
        if max(max(abs(imag([gx{s};hx{s}])))) < 1e-8       
            gx{s} = real(gx{s});
            hx{s} = real(hx{s});
        else
            warning('Large Imaginary Portions of Decision Rules');
            gx{s} = Inf*ones(ny,nx);
            hx{s} = Inf*ones(nx,nx);
            stop = 1;
        end
        
        % increment s
        s = s + 1;
    end 
    
    % check norm if solved correctly, otherwise terminate loop
    if stop == 0
        norm = 0;
        for s = 1:ns
            norm = max([norm;max(max(abs([gx{s}-gx0{s};hx{s}-hx0{s}])))]);
        end
    else
        norm = 0;
        jj = maxit;
    end
end

% -- Check for MSS -- %
if stop == 1
    retcode = 0;
    disp(['Iterative procedure did not converge, norm = ' num2str(norm)]);
else
    % Constructing MSS Check Matrix 
    C = kron(Pmatss',eye(nx^2,nx^2));
    N = zeros(ns*nx*nx,ns*nx*nx);
    for ii = 1:ns
        N((ii-1)*nx*nx+1:ii*nx*nx,(ii-1)*nx*nx+1:ii*nx*nx) ...
            = kron(hx{ii},hx{ii});
    end
    T = C*N;
    maxeig = max(abs(eig(T)));
    
    % Converged to MSS solution
    if maxeig < 1 && jj < maxit
        disp(['Iterative procedure converged to MSS solution after ' num2str(jj) ' iterations, norm = ' num2str(norm)]);
        retcode = 1;
    % Converged to non MSS solution
    elseif maxeig >= 1 && jj < maxit
        disp(['Iterative procedure converged fter ' num2str(jj) ' iterations, but not MSS solution, norm = ' num2str(norm)]);
        retcode = 0;
    % Non-convergence to MSS solution
    elseif maxeig < 1 && jj == maxit
        disp(['Iterative procedure hit maximum of ' num2str(maxit) ' iterations, resulting solution is MSS, norm = ' num2str(norm)]);
        retcode = 1;
    else
        disp(['Iterative procedure hit maximum of ' num2str(maxit) ' iterations, resulting solution is not MSS, norm = ' num2str(norm)]);
        retcode = 0;
    end
end


    